// background.js for sumo-analyzer-cors-helper
// Responds to CORS_HELPER_PING from content script
// Sets extension badge based on availability for the active tab and allowed hosts

// Build allowed hosts from manifest content_scripts.matches
let ALLOWED_HOSTS = [];
function extractHostFromPattern(pattern) {
  try {
    // Remove wildcards and trailing slashes, get host:port
    const cleaned = pattern.replace(/\*/g, '').replace(/\/\*$/, '');
    const url = new URL(cleaned);
    return url.host;
  } catch {
    return null;
  }
}

function updateAllowedHosts() {
  const manifest = chrome.runtime.getManifest();
  const hosts = new Set();
  if (manifest.content_scripts) {
    manifest.content_scripts.forEach(cs => {
      (cs.matches || []).forEach(pattern => {
        const host = extractHostFromPattern(pattern);
        if (host) hosts.add(host);
      });
    });
  }
  ALLOWED_HOSTS = Array.from(hosts);
}

updateAllowedHosts();

function isAllowedHost(url) {
  try {
    return ALLOWED_HOSTS.includes(new URL(url).host);
  } catch {
    return false;
  }
}

function setBadge(tabId, status) {
  chrome.action.setBadgeText({ tabId, text: status ? 'ON' : 'OFF' });
  if (status) {
    chrome.action.setBadgeBackgroundColor({ tabId, color: '#22c55e' }); // green
    chrome.action.setBadgeTextColor &&
      chrome.action.setBadgeTextColor({ tabId, color: '#fff' });
  } else {
    chrome.action.setBadgeBackgroundColor({ tabId, color: '#ffe066' }); // soft yellow
    chrome.action.setBadgeTextColor &&
      chrome.action.setBadgeTextColor({ tabId, color: '#222' });
  }
}

// Listen for messages from content script
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request && request.type === 'CORS_HELPER_PING') {
    if (sender.tab && sender.tab.id !== undefined) {
      const setOn = url => setBadge(sender.tab.id, url && isAllowedHost(url));
      if (sender.tab.url) {
        setOn(sender.tab.url);
      } else {
        chrome.tabs.get(sender.tab.id, tab => setOn(tab?.url));
      }
    }
    sendResponse({ available: true });
  }
});

// Set badge to OFF by default when tab is updated or activated (unless allowed)
function handleTab(tabId) {
  chrome.tabs.get(tabId, tab => {
    if (!tab?.url || !isAllowedHost(tab.url)) {
      setBadge(tabId, false);
    }
    // If allowed, do nothing: wait for content script ping to set ON
  });
}

chrome.tabs.onActivated.addListener(activeInfo => handleTab(activeInfo.tabId));
chrome.tabs.onUpdated.addListener((tabId, changeInfo) => {
  if (changeInfo.status === 'loading') handleTab(tabId);
});

// Debug logging for rule matching (only available in development mode)
// Check if the API is available before using it
if (
  chrome.declarativeNetRequest &&
  chrome.declarativeNetRequest.onRuleMatchedDebug
) {
  chrome.declarativeNetRequest.onRuleMatchedDebug.addListener(details => {
    console.log('CORS Helper - Rule matched:', details);
  });
} else {
  console.log(
    'CORS Helper - Debug mode not available (this is normal in production)',
  );
}
